#ifndef MOD_NVG_HLSL
#define MOD_NVG_HLSL

////////////////////////////////////////////////////////////////////////////////////////
////// USER DEFINED VARS - CHANGE TO EFFECT SHAPE, SIZE AND CENTER POSITION OF NVG /////

//overall NVG postion and size
static const float2 EYE_CENTER = float2(0.5f, 0.6f);  // SETS CENTER POINT OF NVG EFFECT ON SCREEN
static const float2 EYE_SIZE = float2(0.45f, 0.8f); // SETS X AND Y DIAMETER OF NVG (PERCENTAGE OF SCREEN)

//sets which NVG type to use. 0 = full screen, 1 = NVG with no ring, 2 = NVG with Ring, 3 = NVG with ring and slight zoom (not functional)
//note: if using render mode = 1, nvg will only be size of RInner below. ie RInner(percentage) * Eye_Size = final NVG size
static const int RenderMode = 2;

//RING Sizing
static const float ROFade = 0.90f;
static const float RMerge = 0.65f;
static const float RInner = 0.63f; 

// This value and the NVG intensity control (in game) controls the amount of gain of the NVG. 
// Larger value means NVG is brighter. RECOMMENDED AT 0.10f and 0.08f
static const float NVG_LIFT[2] =
	{  	0.10f, // A-10C
		0.08f  // KA-50
	};

// Value controls intensity of noise as gain is increased. Larger value means less noise. Note that Russian NVG has more noise coded into it so value should be higher.
// RECOMMENDED AT 2 and 2
static const float NVG_NOISE[2] =
	{  	2, // A-10C
		2  // KA-50
	};


///////////////////////////////////////////////////////////////////////////////
////// STATIC VARS - DONT CHANGE UNLESS YOU HAVE AN UNDERSTANDING OF CODE /////

// NVG Type 0 = A-10C. 1 = KA-50.  /////DO NOT CHANGE. PERSISTENT THROUGH MULTIPLE FILES
static const int NVG_A10 = 0;	
static const int NVG_KA50 = 1;  

// Min value to multiply noise.
static const float NOISE_MIN[2] =
	{ 	0.5f, // A-10C
		0.75f // KA-50
	};

// Max value to multiply noise.
static const float NOISE_MAX[2] =
	{	0.5f, // A-10C
		0.75f // KA-50
	};

// Color to use for NVGs.
static const float2x3 COLORIZE_COLOR =
	{	{0.65f, 1.00f, 0.35f}, // A-10C
		{0.2f, 0.46f, 0.22f}   // KA-50
	};


	
////////////////////////////////////
////////// PROCEDURES //////////////

///////// CONVERT RGB TO HSL
float3 RGBtoHSL(const float3 rgb): COLOR0
{
  float r = clamp(rgb.r, 0.0f, 1.0f);
  float g = clamp(rgb.g, 0.0f, 1.0f);
  float b = clamp(rgb.b, 0.0f, 1.0f);
  float maxV = max(max(r, g), b);
  float minV = min(min(r, g), b);
  float h = (maxV + minV) / 2.0f;
  float s = h;
  float l = h;
  if (maxV == minV)
	{ h = 0.0f;
	  s = 0.0f;
	}
  else
	{float d = maxV - minV;
		if (l > 0.5f)
			{ s = d / (2.0f - maxV - minV);
			}
		else
			{ s = d / (maxV + minV);
			}
		if (maxV == r)
			{ h = (g - b) / d;
			}
		else
			{ 
			if (maxV = g)
				{ h = ((b - r) / d) + 2.0f;
				}
			else
				{ h = ((r - g) / d) + 4.0f;
				}
			}
		h /= 6.0f;
	}
return float3(h, s, l);
}

///// CONVERT HUE VALUE SINGLE RGB VALUE, USED IN HSL TO RBG
float hue2rgb(const float p, const float q, const float t)
{
  float T = t;
  if (t < 0.0f)
	{ T += 1.0f;
	}
  if (t > 1.0f)
	{ T -= 1.0f;
	}
  if (T < 1.0f / 6.0f)
	{ return p + (q - p) * 6.0f * T;
	}
  if (T < 1.0f / 2.0f)
	{ return q;
	}
  if (T < 2.0f / 3.0f)
	{return p + (q - p) * (2.0 / 3.0f - T) * 6.0f;
	}
  return p;
}

///// HSL TO RGB COLOR CONVERSION
float3 HSLtoRGB(const float3 hsl): COLOR0
{
  const float h = hsl.r;
  const float s = hsl.g;
  const float l = hsl.b;
  float r = 0.0f;
  float g = 0.0f;
  float b = 0.0f;
  if (s == 0.0f)
  { 
	r = 1.0f;
    g = 1.0f;
    b = 1.0f;
  }
  else
  {
    float q = l < 0.5f ? l * (1 + s) : l + s - l * s;
    float p = 2 * l - q;
    r = hue2rgb(p, q, h + (1.0f / 3.0f));
    g = hue2rgb(p, q, h);
    b = hue2rgb(p, q, h - (1.0f / 3.0f));
  }
return float3(r, g, b);
}

////////// CREATE COLOR TINGE - Uses COLORIZE_COLOR defined in static constants depending on aircraft
float3 colorize(const float3 rgb, const float3 to_color): COLOR0
{
  const float3 hsl = RGBtoHSL(to_color);
  const float maxV = max(max(rgb.r, rgb.g), rgb.b);
  const float minV = min(min(rgb.r, rgb.g), rgb.b);
  const float lum = (maxV + minV) / 2.0f;

  if (lum > 1.0f)
  {
    return float3(1.0f, 1.0f, 1.0f);
  }

  return HSLtoRGB(float3(hsl.r, hsl.g, lum));
}

float3 maskNVG(const float2 uv): SV_TARGET0
{	
	const float ESRx = pow((EYE_SIZE.x)/2, 2);  //radius of NVG - x axis
	const float ESRy = pow((EYE_SIZE.y)/2, 2);  //radius of NVG - y axis
	float posx = uv.x - EYE_CENTER.x; 			//pixel distance from nvg center - x axis
	float posy = uv.y - (1-EYE_CENTER.y);		//pixel distance from nvg center - y axis
	float distance = (pow(posx, 2)/(ESRx)) + (pow(posy, 2)/(ESRy));  //pixel distance from nvg center - radial
	float x = 0.0f;		//used for mask
	float y = 0.0f;		//used for mask
	float z = 0.0f;		//used for mask
	
	{if (distance > 1)
		{return float3(x, y, z);
			}
	else
		{if (distance > ROFade)
			{ 
			z = 1 - smoothstep(ROFade, 1.0, distance);
			y = 1.0f;
			x = 0.0f;
			return float3(x, y, z);
			}
		else if (distance > RMerge)
			{
			z = 1.0f;
			y = 1.0f;
			x = 0.0f;
			return float3(x, y, z);
			}
		else if (distance > RInner)
			{
			z = 1.0f;
			y = smoothstep(RInner, RMerge, distance);
			x = 1.0f - y;
			return float3(x, y, z);
			}
		else 
			{
			z = 1.0f;
			y = 0.0f;
			x = 1.0f;
			return float3(x, y, z);
			}
		}
	}
}	

#endif